/**
 * ParcelPulse Bridge - Popup Script
 *
 * Mostra stato delle sessioni catturate e azioni disponibili.
 */

// URL della webapp (development e production)
const WEBAPP_URLS = [
  "http://localhost:3001/it/dashboard",
  "https://parcelpulse.app/it/dashboard",
];

// Flag per dominio Amazon
const DOMAIN_FLAGS = {
  "www.amazon.it": "🇮🇹",
  "www.amazon.com": "🇺🇸",
  "www.amazon.co.uk": "🇬🇧",
  "www.amazon.de": "🇩🇪",
  "www.amazon.fr": "🇫🇷",
  "www.amazon.es": "🇪🇸",
  "www.amazon.co.jp": "🇯🇵",
  "www.amazon.in": "🇮🇳",
  "www.amazon.com.au": "🇦🇺",
  "www.amazon.ca": "🇨🇦",
  "www.amazon.com.br": "🇧🇷",
  "www.amazon.com.sg": "🇸🇬",
  "www.amazon.nl": "🇳🇱",
  "www.amazon.pl": "🇵🇱",
  "www.amazon.se": "🇸🇪",
  "www.amazon.ae": "🇦🇪",
  "www.amazon.sa": "🇸🇦",
  "www.amazon.com.mx": "🇲🇽",
  "www.amazon.com.tr": "🇹🇷",
  "www.amazon.com.be": "🇧🇪",
  "www.amazon.eg": "🇪🇬",
  "www.amazon.co": "🇨🇴",
};

function timeAgo(timestamp) {
  if (!timestamp) return "mai";
  const diff = Date.now() - timestamp;
  const minutes = Math.floor(diff / 60000);
  if (minutes < 1) return "ora";
  if (minutes < 60) return `${minutes}m fa`;
  const hours = Math.floor(minutes / 60);
  if (hours < 24) return `${hours}h fa`;
  return `${Math.floor(hours / 24)}g fa`;
}

async function loadStatus() {
  const status = await chrome.runtime.sendMessage({ action: "getStatus" });

  // Version
  document.getElementById("version").textContent = `v${status.version}`;

  // Status indicator
  const dot = document.getElementById("statusDot");
  const text = document.getElementById("statusText");

  if (status.connected) {
    dot.className = "dot connected";
    const count = status.domains.length;
    text.textContent = `${count} session${count > 1 ? "i" : "e"} attiv${count > 1 ? "e" : "a"}`;
  } else {
    dot.className = "dot disconnected";
    text.textContent = "Nessuna sessione attiva";
  }

  // Sessions list
  renderSessions(status.sessions);
}

function renderSessions(sessions) {
  const container = document.getElementById("sessionsList");

  if (!sessions || Object.keys(sessions).length === 0) {
    container.innerHTML = `
      <div class="empty-state">
        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" class="empty-icon">
          <circle cx="11" cy="11" r="8"/>
          <path d="m21 21-4.3-4.3"/>
        </svg>
        <p>Nessuna sessione. Visita Amazon per catturare automaticamente il token.</p>
      </div>
    `;
    return;
  }

  const sorted = Object.values(sessions).sort(
    (a, b) => (b.updatedAt || 0) - (a.updatedAt || 0)
  );

  container.innerHTML = sorted
    .map(
      (s) => `
    <div class="session-item" data-domain="${s.domain}">
      <div class="session-info">
        <div class="session-domain">
          <span class="flag">${DOMAIN_FLAGS[s.domain] || "🌐"}</span>
          ${s.domain.replace("www.", "")}
        </div>
        <div class="session-meta">
          <span class="token-badge">Token</span>
          &middot; ${timeAgo(s.updatedAt)}
        </div>
      </div>
      <div class="session-actions">
        <button class="btn-icon refresh-btn" title="Aggiorna" data-domain="${s.domain}">
          <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <path d="M21 12a9 9 0 0 0-9-9 9.75 9.75 0 0 0-6.74 2.74L3 8"/>
            <path d="M3 3v5h5"/>
            <path d="M3 12a9 9 0 0 0 9 9 9.75 9.75 0 0 0 6.74-2.74L21 16"/>
            <path d="M16 16h5v5"/>
          </svg>
        </button>
        <button class="btn-icon danger delete-btn" title="Rimuovi" data-domain="${s.domain}">
          <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <path d="M18 6 6 18"/><path d="m6 6 12 12"/>
          </svg>
        </button>
      </div>
    </div>
  `
    )
    .join("");

  // Event listeners
  container.querySelectorAll(".refresh-btn").forEach((btn) => {
    btn.addEventListener("click", async (e) => {
      e.stopPropagation();
      const domain = btn.dataset.domain;
      btn.style.opacity = "0.5";
      await chrome.runtime.sendMessage({ action: "forceRefresh", domain });
      await loadStatus();
    });
  });

  container.querySelectorAll(".delete-btn").forEach((btn) => {
    btn.addEventListener("click", async (e) => {
      e.stopPropagation();
      const domain = btn.dataset.domain;
      await chrome.runtime.sendMessage({ action: "clearSession", domain });
      await loadStatus();
    });
  });
}

// Open webapp
document.getElementById("openWebapp").addEventListener("click", async () => {
  // Prova prima localhost (development), poi produzione
  const tabs = await chrome.tabs.query({});
  const existingTab = tabs.find((t) =>
    WEBAPP_URLS.some((url) => t.url?.startsWith(url.split("/it/")[0]))
  );

  if (existingTab) {
    chrome.tabs.update(existingTab.id, { active: true });
    chrome.windows.update(existingTab.windowId, { focused: true });
  } else {
    chrome.tabs.create({ url: WEBAPP_URLS[0] });
  }
  window.close();
});

// Clear all
document.getElementById("clearAll").addEventListener("click", async () => {
  await chrome.runtime.sendMessage({ action: "clearAll" });
  await loadStatus();
});

// ============================================
// Tracking IDs rilevati
// ============================================

async function loadTrackingIds() {
  const result = await chrome.storage.local.get("detectedTrackingIds");
  const ids = result.detectedTrackingIds || {};
  const section = document.getElementById("trackingSection");
  const container = document.getElementById("trackingList");

  if (Object.keys(ids).length === 0) {
    section.style.display = "none";
    return;
  }

  section.style.display = "block";

  const sorted = Object.entries(ids).sort(
    ([, a], [, b]) => (b.detectedAt || 0) - (a.detectedAt || 0)
  );

  container.innerHTML = sorted
    .map(
      ([id, data]) => `
    <div class="session-item tracking-item">
      <div class="session-info">
        <div class="session-domain" style="font-family: monospace; font-size: 11px;">
          <span class="flag">${DOMAIN_FLAGS[data.domain] || "📦"}</span>
          ${id}
        </div>
        <div class="session-meta">
          ${timeAgo(data.detectedAt)}
        </div>
      </div>
      <div class="session-actions">
        <button class="btn-icon import-tracking-btn" title="Importa in ParcelPulse" data-id="${id}" data-domain="${data.domain}">
          <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#06B6D4" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <path d="M12 5v14"/><path d="m19 12-7 7-7-7"/>
          </svg>
        </button>
      </div>
    </div>
  `
    )
    .join("");

  // Import click
  container.querySelectorAll(".import-tracking-btn").forEach((btn) => {
    btn.addEventListener("click", async () => {
      const trackingId = btn.dataset.id;
      const btnDomain = btn.dataset.domain;

      // Salva come pending import
      await chrome.runtime.sendMessage({
        action: "pendingImport",
        trackingIds: [trackingId],
        domain: btnDomain,
      });

      // Apri webapp
      const webappUrl = `http://localhost:3001/it/dashboard?import=${trackingId}`;
      const tabs = await chrome.tabs.query({});
      const existingTab = tabs.find((t) =>
        WEBAPP_URLS.some((url) => t.url?.startsWith(url.split("/it/")[0]))
      );

      if (existingTab) {
        chrome.tabs.update(existingTab.id, { active: true, url: webappUrl });
        chrome.windows.update(existingTab.windowId, { focused: true });
      } else {
        chrome.tabs.create({ url: webappUrl });
      }
      window.close();
    });
  });
}

// Init
loadStatus();
loadTrackingIds();
